<?php
/**
 * Funciones auxiliares para el sistema PDV
 */

// Iniciar la sesión sólo si aún no está iniciada
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

/**
 * Carga la configuración desde el archivo config.php.
 *
 * @return array|null Devuelve la configuración o null si no existe.
 */
function loadConfig(): ?array
{
    $configFile = __DIR__ . '/config.php';
    if (file_exists($configFile)) {
        return require $configFile;
    }
    return null;
}

/**
 * Devuelve una conexión PDO a la base de datos del PDV.
 *
 * @return PDO|null
 */
function getPDO(): ?PDO
{
    static $pdo;
    if ($pdo instanceof PDO) {
        return $pdo;
    }
    $config = loadConfig();
    if (!$config || empty($config['pdv'])) {
        return null;
    }
    $db = $config['pdv'];
    $dsn = sprintf('mysql:host=%s;dbname=%s;charset=%s', $db['host'], $db['database'], $db['charset'] ?? 'utf8mb4');
    try {
        $pdo = new PDO($dsn, $db['username'], $db['password'], [
            PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES   => false,
        ]);
        return $pdo;
    } catch (PDOException $e) {
        // No exponer detalles sensibles
        return null;
    }
}

/**
 * Escapa una cadena para mostrarla en HTML.
 *
 * @param string|null $str
 * @return string
 */
function escape(?string $str): string
{
    return htmlspecialchars($str ?? '', ENT_QUOTES, 'UTF-8');
}

/**
 * Redirige a la URL indicada y detiene la ejecución.
 *
 * @param string $url
 */
function redirect(string $url): void
{
    header('Location: ' . $url);
    exit;
}

/**
 * Genera una cadena aleatoria de la longitud especificada.
 *
 * @param int $length
 * @return string
 */
function randomString(int $length = 32): string
{
    return bin2hex(random_bytes($length / 2));
}

/**
 * Devuelve la URL base de la aplicación.
 *
 * Si se ha configurado `app.base_url` se usará ese valor. En caso contrario,
 * se construirá a partir de la URL actual sin la parte de la ruta del script.
 *
 * @param string $path Ruta opcional para añadir a la base.
 * @return string
 */
function baseUrl(string $path = ''): string
{
    $config = loadConfig();
    $base = '';
    if ($config && !empty($config['app']['base_url'])) {
        $base = rtrim($config['app']['base_url'], '/');
    } else {
        // Construir base URL a partir de la petición
        $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        // Directorio del script actual
        $scriptDir = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/');
        $base = $protocol . '://' . $host . $scriptDir;
    }
    if ($path) {
        return $base . '/' . ltrim($path, '/');
    }
    return $base;
}

/**
 * Comprueba si el usuario está logueado.
 *
 * @return bool
 */
function isLoggedIn(): bool
{
    return isset($_SESSION['user_id']);
}

/**
 * Obliga a que el usuario esté logueado. Si no lo está, redirige a login.php.
 */
function requireLogin(): void
{
    if (!isLoggedIn()) {
        redirect('login.php');
    }
}

/**
 * Devuelve la información del usuario actualmente logueado.
 *
 * @return array|null
 */
function currentUser(): ?array
{
    if (!isLoggedIn()) {
        return null;
    }
    $pdo = getPDO();
    if (!$pdo) {
        return null;
    }
    $stmt = $pdo->prepare('SELECT * FROM users WHERE id = :id');
    $stmt->execute([':id' => $_SESSION['user_id']]);
    return $stmt->fetch() ?: null;
}

/**
 * Crea las tablas necesarias para el PDV en la base de datos.
 *
 * @param PDO $pdo
 */
function createTables(PDO $pdo): void
{
    // Tabla de usuarios
    $pdo->exec(
        "CREATE TABLE IF NOT EXISTS users (\n" .
        "    id INT AUTO_INCREMENT PRIMARY KEY,\n" .
        "    username VARCHAR(100) NOT NULL UNIQUE,\n" .
        "    password VARCHAR(255) NOT NULL,\n" .
        "    role VARCHAR(50) NOT NULL DEFAULT 'superadmin',\n" .
        "    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP\n" .
        ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;"
    );

    // Tabla de configuraciones (clave/valor)
    $pdo->exec(
        "CREATE TABLE IF NOT EXISTS settings (\n" .
        "    id INT AUTO_INCREMENT PRIMARY KEY,\n" .
        "    name VARCHAR(100) NOT NULL UNIQUE,\n" .
        "    value TEXT NOT NULL\n" .
        ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;"
    );
}

/**
 * Inserta el usuario SuperAdmin en la base de datos.
 *
 * @param PDO $pdo
 * @param string $username
 * @param string $password
 */
function insertSuperAdmin(PDO $pdo, string $username, string $password): void
{
    $hash = password_hash($password, PASSWORD_DEFAULT);
    $stmt = $pdo->prepare('INSERT INTO users (username, password, role) VALUES (:u, :p, :r)');
    $stmt->execute([
        ':u' => $username,
        ':p' => $hash,
        ':r' => 'superadmin',
    ]);
}

/**
 * Guarda una clave de configuración en la tabla settings. Si ya existe, la actualiza.
 *
 * @param PDO $pdo
 * @param string $name
 * @param string $value
 */
function saveSetting(PDO $pdo, string $name, string $value): void
{
    $stmt = $pdo->prepare('INSERT INTO settings (name, value) VALUES (:n, :v) ON DUPLICATE KEY UPDATE value = VALUES(value)');
    $stmt->execute([
        ':n' => $name,
        ':v' => $value,
    ]);
}

/**
 * Obtiene el valor de una configuración por su nombre.
 *
 * @param PDO $pdo
 * @param string $name
 * @return string|null
 */
function getSetting(PDO $pdo, string $name): ?string
{
    $stmt = $pdo->prepare('SELECT value FROM settings WHERE name = :n');
    $stmt->execute([':n' => $name]);
    $row = $stmt->fetch();
    return $row['value'] ?? null;
}

/**
 * Guarda el archivo de configuración con los parámetros proporcionados.
 *
 * @param array $config
 * @param string $path
 * @throws RuntimeException
 */
function saveConfigFile(array $config, string $path): void
{
    $export = var_export($config, true);
    $content = "<?php\nreturn " . $export . ";\n";
    if (file_put_contents($path, $content) === false) {
        throw new RuntimeException('No se pudo guardar el archivo de configuración.');
    }
    // Proteger el archivo contra lectura por el navegador (requiere .htaccess en servidores Apache)
}