<?php
/**
 * Asistente de instalación para el sistema PDV.
 *
 * Este script guía al usuario a través de los pasos necesarios para
 * configurar la conexión a la base de datos, crear tablas y generar
 * el archivo de configuración. Una vez instalado, la carpeta `install`
 * debería eliminarse o restringirse.
 */

require_once __DIR__ . '/../includes/functions.php';

$configFile = __DIR__ . '/../includes/config.php';

// Si ya existe configuración, no permitir reinstalación
if (file_exists($configFile)) {
    echo '<!DOCTYPE html><html lang="es"><head><meta charset="utf-8"><title>Sistema instalado</title></head><body>';
    echo '<p>El sistema ya está instalado. <a href="../index.php">Ir al inicio</a>.</p>';
    echo '</body></html>';
    exit;
}

$errors = [];
$success = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Recoger datos del formulario
    $pdvHost     = trim($_POST['pdv_host'] ?? '');
    $pdvDb       = trim($_POST['pdv_db'] ?? '');
    $pdvUser     = trim($_POST['pdv_user'] ?? '');
    $pdvPass     = $_POST['pdv_pass'] ?? '';

    $psHost      = trim($_POST['ps_host'] ?? '');
    $psDb        = trim($_POST['ps_db'] ?? '');
    $psUser      = trim($_POST['ps_user'] ?? '');
    $psPass      = $_POST['ps_pass'] ?? '';
    $psPrefix    = trim($_POST['ps_prefix'] ?? 'ps_');

    $adminUser   = trim($_POST['admin_user'] ?? '');
    $adminPass   = $_POST['admin_pass'] ?? '';
    $adminPass2  = $_POST['admin_pass2'] ?? '';
    $baseUrl     = trim($_POST['base_url'] ?? '');

    // Validaciones básicas
    if ($pdvHost === '' || $pdvDb === '' || $pdvUser === '' || $adminUser === '' || $adminPass === '') {
        $errors[] = 'Todos los campos obligatorios deben completarse.';
    }
    if ($adminPass !== $adminPass2) {
        $errors[] = 'Las contraseñas no coinciden.';
    }

    if (empty($errors)) {
        // Intentar conexión a la BD del PDV
        $dsn = sprintf('mysql:host=%s;dbname=%s;charset=utf8mb4', $pdvHost, $pdvDb);
        try {
            $pdo = new PDO($dsn, $pdvUser, $pdvPass, [
                PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES   => false,
            ]);

            // Crear tablas iniciales
            createTables($pdo);

            // Insertar usuario SuperAdmin
            insertSuperAdmin($pdo, $adminUser, $adminPass);

            // Crear tablas de roles y permisos y asignar SuperAdmin
            ensureRoleTables($pdo);

            // Guardar datos de conexión a PrestaShop en tabla settings
            saveSetting($pdo, 'ps_host', $psHost);
            saveSetting($pdo, 'ps_db', $psDb);
            saveSetting($pdo, 'ps_user', $psUser);
            saveSetting($pdo, 'ps_pass', $psPass);
            saveSetting($pdo, 'ps_prefix', $psPrefix);

            // Crear archivo de configuración
            $config = [
                'pdv' => [
                    'host'     => $pdvHost,
                    'database' => $pdvDb,
                    'username' => $pdvUser,
                    'password' => $pdvPass,
                    'charset'  => 'utf8mb4',
                ],
                'prestashop' => [
                    'host'     => $psHost,
                    'database' => $psDb,
                    'username' => $psUser,
                    'password' => $psPass,
                    'prefix'   => $psPrefix ?: 'ps_',
                    'charset'  => 'utf8mb4',
                ],
                'app' => [
                    'secret_key' => randomString(64),
                    'base_url'   => $baseUrl,
                ],
            ];
            saveConfigFile($config, $configFile);

            // Asegurar tablas para cajas y sesiones
            // Esto crea las tablas `boxes`, `cash_sessions` y `cash_movements` para el hito de control de caja.
            ensureBoxTables($pdo);

            $success = true;
        } catch (Exception $e) {
            $errors[] = 'Error al conectar o inicializar la base de datos: ' . $e->getMessage();
        }
    }
}

?><!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Instalación PDV</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet" integrity="sha384-2ZY5sxzobS//s3m4W9Pi1s2ynG1iXGk4MReeQ0bGAP7zRvH+6zCuN6E6a2hZIlK" crossorigin="anonymous">
    <style>
        body {
            background-color: #f8f9fa;
        }
        .install-form {
            max-width: 800px;
            margin: 30px auto;
            padding: 20px;
            background-color: #fff;
            border-radius: 8px;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
        }
    </style>
</head>
<body>
<div class="container">
    <div class="install-form">
        <h2 class="mb-4">Instalación del Sistema PDV</h2>
        <?php if ($success): ?>
            <div class="alert alert-success">
                <p>¡Instalación completada con éxito!</p>
                <p>Ahora puede <a href="../login.php">iniciar sesión</a> con el usuario creado.</p>
            </div>
        <?php else: ?>
            <?php if ($errors): ?>
                <div class="alert alert-danger">
                    <ul class="mb-0">
                        <?php foreach ($errors as $err): ?>
                            <li><?php echo escape($err); ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>
            <form method="post" action="">
                <h5>Base de Datos del PDV</h5>
                <div class="row mb-3">
                    <div class="col-md-6">
                        <label for="pdv_host" class="form-label">Servidor</label>
                        <input type="text" class="form-control" id="pdv_host" name="pdv_host" required value="<?php echo escape($_POST['pdv_host'] ?? 'localhost'); ?>">
                    </div>
                    <div class="col-md-6">
                        <label for="pdv_db" class="form-label">Base de datos</label>
                        <input type="text" class="form-control" id="pdv_db" name="pdv_db" required value="<?php echo escape($_POST['pdv_db'] ?? 'pdv'); ?>">
                    </div>
                </div>
                <div class="row mb-3">
                    <div class="col-md-6">
                        <label for="pdv_user" class="form-label">Usuario</label>
                        <input type="text" class="form-control" id="pdv_user" name="pdv_user" required value="<?php echo escape($_POST['pdv_user'] ?? 'root'); ?>">
                    </div>
                    <div class="col-md-6">
                        <label for="pdv_pass" class="form-label">Contraseña</label>
                        <input type="password" class="form-control" id="pdv_pass" name="pdv_pass" value="<?php echo escape($_POST['pdv_pass'] ?? ''); ?>">
                    </div>
                </div>
                <h5 class="mt-4">Base de Datos de PrestaShop</h5>
                <div class="row mb-3">
                    <div class="col-md-6">
                        <label for="ps_host" class="form-label">Servidor</label>
                        <input type="text" class="form-control" id="ps_host" name="ps_host" required value="<?php echo escape($_POST['ps_host'] ?? 'localhost'); ?>">
                    </div>
                    <div class="col-md-6">
                        <label for="ps_db" class="form-label">Base de datos</label>
                        <input type="text" class="form-control" id="ps_db" name="ps_db" required value="<?php echo escape($_POST['ps_db'] ?? 'prestashop'); ?>">
                    </div>
                </div>
                <div class="row mb-3">
                    <div class="col-md-6">
                        <label for="ps_user" class="form-label">Usuario</label>
                        <input type="text" class="form-control" id="ps_user" name="ps_user" required value="<?php echo escape($_POST['ps_user'] ?? 'root'); ?>">
                    </div>
                    <div class="col-md-6">
                        <label for="ps_pass" class="form-label">Contraseña</label>
                        <input type="password" class="form-control" id="ps_pass" name="ps_pass" value="<?php echo escape($_POST['ps_pass'] ?? ''); ?>">
                    </div>
                </div>
                <div class="mb-3">
                    <label for="ps_prefix" class="form-label">Prefijo de tablas</label>
                    <input type="text" class="form-control" id="ps_prefix" name="ps_prefix" value="<?php echo escape($_POST['ps_prefix'] ?? 'ps_'); ?>">
                    <div class="form-text">Por defecto es "ps_".</div>
                </div>
                <h5 class="mt-4">Usuario SuperAdmin</h5>
                <div class="mb-3">
                    <label for="admin_user" class="form-label">Usuario</label>
                    <input type="text" class="form-control" id="admin_user" name="admin_user" required value="<?php echo escape($_POST['admin_user'] ?? 'admin'); ?>">
                </div>
                <div class="row mb-3">
                    <div class="col-md-6">
                        <label for="admin_pass" class="form-label">Contraseña</label>
                        <input type="password" class="form-control" id="admin_pass" name="admin_pass" required>
                    </div>
                    <div class="col-md-6">
                        <label for="admin_pass2" class="form-label">Repetir contraseña</label>
                        <input type="password" class="form-control" id="admin_pass2" name="admin_pass2" required>
                    </div>
                </div>
                <h5 class="mt-4">Configuración general</h5>
                <div class="mb-3">
                    <label for="base_url" class="form-label">URL base (opcional)</label>
                    <input type="text" class="form-control" id="base_url" name="base_url" value="<?php echo escape($_POST['base_url'] ?? ''); ?>">
                    <div class="form-text">En blanco para detectar automáticamente.</div>
                </div>
                <button type="submit" class="btn btn-primary">Instalar</button>
            </form>
        <?php endif; ?>
    </div>
</div>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js" integrity="sha384-qo1NK9vdN2AyJEaCmxEtdlJ6dNVY4s9dL7LTrO2TV4QZk2aIy08eRUKqT1syeZye" crossorigin="anonymous"></script>
</body>
</html>