<?php
/**
 * Funciones auxiliares para el sistema PDV
 */

// Iniciar la sesión sólo si aún no está iniciada
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

/**
 * Carga la configuración desde el archivo config.php.
 *
 * @return array|null Devuelve la configuración o null si no existe.
 */
function loadConfig(): ?array
{
    $configFile = __DIR__ . '/config.php';
    if (file_exists($configFile)) {
        return require $configFile;
    }
    return null;
}

/**
 * Devuelve una conexión PDO a la base de datos del PDV.
 *
 * @return PDO|null
 */
function getPDO(): ?PDO
{
    static $pdo;
    if ($pdo instanceof PDO) {
        return $pdo;
    }
    $config = loadConfig();
    if (!$config || empty($config['pdv'])) {
        return null;
    }
    $db = $config['pdv'];
    $dsn = sprintf('mysql:host=%s;dbname=%s;charset=%s', $db['host'], $db['database'], $db['charset'] ?? 'utf8mb4');
    try {
        $pdo = new PDO($dsn, $db['username'], $db['password'], [
            PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES   => false,
        ]);
        return $pdo;
    } catch (PDOException $e) {
        // No exponer detalles sensibles
        return null;
    }
}

/**
 * Escapa una cadena para mostrarla en HTML.
 *
 * @param string|null $str
 * @return string
 */
function escape(?string $str): string
{
    return htmlspecialchars($str ?? '', ENT_QUOTES, 'UTF-8');
}

/**
 * Redirige a la URL indicada y detiene la ejecución.
 *
 * @param string $url
 */
function redirect(string $url): void
{
    header('Location: ' . $url);
    exit;
}

/**
 * Genera una cadena aleatoria de la longitud especificada.
 *
 * @param int $length
 * @return string
 */
function randomString(int $length = 32): string
{
    return bin2hex(random_bytes($length / 2));
}

/**
 * Devuelve la URL base de la aplicación.
 *
 * Si se ha configurado `app.base_url` se usará ese valor. En caso contrario,
 * se construirá a partir de la URL actual sin la parte de la ruta del script.
 *
 * @param string $path Ruta opcional para añadir a la base.
 * @return string
 */
function baseUrl(string $path = ''): string
{
    $config = loadConfig();
    $base = '';
    if ($config && !empty($config['app']['base_url'])) {
        $base = rtrim($config['app']['base_url'], '/');
    } else {
        // Construir base URL a partir de la petición
        $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        // Directorio del script actual
        $scriptDir = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/');
        $base = $protocol . '://' . $host . $scriptDir;
    }
    if ($path) {
        return $base . '/' . ltrim($path, '/');
    }
    return $base;
}

/**
 * Comprueba si el usuario está logueado.
 *
 * @return bool
 */
function isLoggedIn(): bool
{
    return isset($_SESSION['user_id']);
}

/**
 * Obliga a que el usuario esté logueado. Si no lo está, redirige a login.php.
 */
function requireLogin(): void
{
    if (!isLoggedIn()) {
        redirect('login.php');
    }
}

/**
 * Devuelve la información del usuario actualmente logueado.
 *
 * @return array|null
 */
function currentUser(): ?array
{
    if (!isLoggedIn()) {
        return null;
    }
    $pdo = getPDO();
    if (!$pdo) {
        return null;
    }
    $stmt = $pdo->prepare('SELECT * FROM users WHERE id = :id');
    $stmt->execute([':id' => $_SESSION['user_id']]);
    return $stmt->fetch() ?: null;
}

/**
 * Crea las tablas necesarias para el PDV en la base de datos.
 *
 * @param PDO $pdo
 */
function createTables(PDO $pdo): void
{
    // Tabla de usuarios
    $pdo->exec(
        "CREATE TABLE IF NOT EXISTS users (\n" .
        "    id INT AUTO_INCREMENT PRIMARY KEY,\n" .
        "    username VARCHAR(100) NOT NULL UNIQUE,\n" .
        "    password VARCHAR(255) NOT NULL,\n" .
        "    role VARCHAR(50) NOT NULL DEFAULT 'superadmin',\n" .
        "    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP\n" .
        ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;"
    );

    // Tabla de configuraciones (clave/valor)
    $pdo->exec(
        "CREATE TABLE IF NOT EXISTS settings (\n" .
        "    id INT AUTO_INCREMENT PRIMARY KEY,\n" .
        "    name VARCHAR(100) NOT NULL UNIQUE,\n" .
        "    value TEXT NOT NULL\n" .
        ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;"
    );
}

/**
 * Crea las tablas necesarias para roles y permisos si no existen y ajusta la tabla de usuarios.
 *
 * @param PDO $pdo
 */
function ensureRoleTables(PDO $pdo): void
{
    // Tabla de roles
    $pdo->exec(
        "CREATE TABLE IF NOT EXISTS roles (\n" .
        "    id INT AUTO_INCREMENT PRIMARY KEY,\n" .
        "    name VARCHAR(100) NOT NULL UNIQUE,\n" .
        "    description VARCHAR(255) NULL,\n" .
        "    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP\n" .
        ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;"
    );

    // Tabla de permisos por rol y módulo
    $pdo->exec(
        "CREATE TABLE IF NOT EXISTS role_permissions (\n" .
        "    id INT AUTO_INCREMENT PRIMARY KEY,\n" .
        "    role_id INT NOT NULL,\n" .
        "    module VARCHAR(100) NOT NULL,\n" .
        "    can_view TINYINT(1) DEFAULT 0,\n" .
        "    can_create TINYINT(1) DEFAULT 0,\n" .
        "    can_edit TINYINT(1) DEFAULT 0,\n" .
        "    can_delete TINYINT(1) DEFAULT 0,\n" .
        "    UNIQUE KEY role_module (role_id, module),\n" .
        "    CONSTRAINT fk_rp_role FOREIGN KEY (role_id) REFERENCES roles(id) ON DELETE CASCADE\n" .
        ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;"
    );

    // Añadir columna role_id a la tabla de usuarios si no existe
    $cols = $pdo->query("SHOW COLUMNS FROM users LIKE 'role_id'")->fetch();
    if (!$cols) {
        $pdo->exec("ALTER TABLE users ADD COLUMN role_id INT NULL AFTER role");
    }

    // Asegurar el rol SuperAdmin
    $stmt = $pdo->prepare('SELECT id FROM roles WHERE name = :name');
    $stmt->execute([':name' => 'SuperAdmin']);
    $superRoleId = $stmt->fetchColumn();
    if (!$superRoleId) {
        // Crear rol SuperAdmin con descripción
        $pdo->prepare('INSERT INTO roles (name, description) VALUES (:name, :desc)')->execute([
            ':name' => 'SuperAdmin',
            ':desc' => 'Rol con permisos completos en el sistema',
        ]);
        $superRoleId = (int)$pdo->lastInsertId();
    }

    // Asignar rol SuperAdmin a usuarios con campo role='superadmin' (valor original)
    $stmt = $pdo->prepare('UPDATE users SET role_id = :role_id WHERE LOWER(role) = :super AND (role_id IS NULL OR role_id = 0)');
    $stmt->execute([
        ':role_id' => $superRoleId,
        ':super'   => 'superadmin',
    ]);

    // Asignar permisos completos al rol SuperAdmin si no existen
    $modules = getModules();
    foreach ($modules as $module) {
        $stmt = $pdo->prepare('SELECT id FROM role_permissions WHERE role_id = :role_id AND module = :module');
        $stmt->execute([
            ':role_id' => $superRoleId,
            ':module'  => $module,
        ]);
        $exists = $stmt->fetch();
        if (!$exists) {
            $pdo->prepare(
                'INSERT INTO role_permissions (role_id, module, can_view, can_create, can_edit, can_delete) VALUES (:role_id, :module, 1, 1, 1, 1)'
            )->execute([
                ':role_id' => $superRoleId,
                ':module'  => $module,
            ]);
        }
    }
}

/**
 * Devuelve la lista de módulos disponibles para permisos.
 *
 * @return string[]
 */
function getModules(): array
{
    return [
        'dashboard',
        'users',
        'roles',
        'boxes',
        'settings',
        'pos',
        'stats',
    ];
}

/**
 * Obtiene todos los roles disponibles.
 *
 * @param PDO $pdo
 * @return array
 */
function getRoles(PDO $pdo): array
{
    $stmt = $pdo->query('SELECT * FROM roles ORDER BY id ASC');
    return $stmt->fetchAll();
}

/**
 * Obtiene la información de un rol por su ID.
 *
 * @param PDO $pdo
 * @param int $roleId
 * @return array|null
 */
function getRoleById(PDO $pdo, int $roleId): ?array
{
    $stmt = $pdo->prepare('SELECT * FROM roles WHERE id = :id');
    $stmt->execute([':id' => $roleId]);
    $role = $stmt->fetch();
    return $role ?: null;
}

/**
 * Crea un rol con sus permisos y devuelve su ID.
 *
 * @param PDO $pdo
 * @param string $name
 * @param string $description
 * @param array $permissions Formato: ['module' => ['view' => bool, 'create' => bool, ...], ...]
 * @return int
 */
function createRole(PDO $pdo, string $name, string $description, array $permissions): int
{
    // Insertar rol
    $stmt = $pdo->prepare('INSERT INTO roles (name, description) VALUES (:name, :description)');
    $stmt->execute([
        ':name'        => $name,
        ':description' => $description,
    ]);
    $roleId = (int)$pdo->lastInsertId();
    // Guardar permisos
    saveRolePermissions($pdo, $roleId, $permissions);
    return $roleId;
}

/**
 * Actualiza un rol y sus permisos.
 *
 * @param PDO $pdo
 * @param int $id
 * @param string $name
 * @param string $description
 * @param array $permissions
 */
function updateRole(PDO $pdo, int $id, string $name, string $description, array $permissions): void
{
    $stmt = $pdo->prepare('UPDATE roles SET name = :name, description = :description WHERE id = :id');
    $stmt->execute([
        ':name'        => $name,
        ':description' => $description,
        ':id'          => $id,
    ]);
    // Actualizar permisos
    saveRolePermissions($pdo, $id, $permissions);
}

/**
 * Elimina un rol y sus permisos.
 * No permite eliminar el rol SuperAdmin.
 *
 * @param PDO $pdo
 * @param int $id
 */
function deleteRole(PDO $pdo, int $id): void
{
    // Comprobar si es SuperAdmin
    $role = getRoleById($pdo, $id);
    if (!$role) {
        return;
    }
    if (strtolower($role['name']) === 'superadmin') {
        return; // No eliminar
    }
    // Eliminar usuarios que tengan este rol: reasignar a null o superadmin
    $stmt = $pdo->prepare('UPDATE users SET role_id = NULL WHERE role_id = :id');
    $stmt->execute([':id' => $id]);
    // Eliminar permisos y rol
    $pdo->prepare('DELETE FROM role_permissions WHERE role_id = :id')->execute([':id' => $id]);
    $pdo->prepare('DELETE FROM roles WHERE id = :id')->execute([':id' => $id]);
}

/**
 * Obtiene los permisos de un rol.
 *
 * @param PDO $pdo
 * @param int $roleId
 * @return array Formato: [module => ['view' => bool, 'create' => bool, 'edit' => bool, 'delete' => bool], ...]
 */
function getRolePermissions(PDO $pdo, int $roleId): array
{
    $modules = getModules();
    $permissions = [];
    foreach ($modules as $module) {
        $permissions[$module] = [
            'view'   => false,
            'create' => false,
            'edit'   => false,
            'delete' => false,
        ];
    }
    $stmt = $pdo->prepare('SELECT module, can_view, can_create, can_edit, can_delete FROM role_permissions WHERE role_id = :role_id');
    $stmt->execute([':role_id' => $roleId]);
    foreach ($stmt as $row) {
        $module = $row['module'];
        $permissions[$module] = [
            'view'   => (bool)$row['can_view'],
            'create' => (bool)$row['can_create'],
            'edit'   => (bool)$row['can_edit'],
            'delete' => (bool)$row['can_delete'],
        ];
    }
    return $permissions;
}

/**
 * Guarda los permisos de un rol.
 *
 * @param PDO $pdo
 * @param int $roleId
 * @param array $permissions
 */
function saveRolePermissions(PDO $pdo, int $roleId, array $permissions): void
{
    // Eliminar permisos existentes
    $stmtDelete = $pdo->prepare('DELETE FROM role_permissions WHERE role_id = :role_id');
    $stmtDelete->execute([':role_id' => $roleId]);
    // Insertar nuevos permisos
    $stmt = $pdo->prepare(
        'INSERT INTO role_permissions (role_id, module, can_view, can_create, can_edit, can_delete) VALUES (:role_id, :module, :view, :create, :edit, :delete)'
    );
    foreach ($permissions as $module => $acts) {
        $stmt->execute([
            ':role_id' => $roleId,
            ':module'  => $module,
            ':view'    => empty($acts['view'])   ? 0 : 1,
            ':create'  => empty($acts['create']) ? 0 : 1,
            ':edit'    => empty($acts['edit'])   ? 0 : 1,
            ':delete'  => empty($acts['delete']) ? 0 : 1,
        ]);
    }
}

/**
 * Obtiene todos los usuarios.
 *
 * @param PDO $pdo
 * @return array
 */
function getUsers(PDO $pdo): array
{
    $stmt = $pdo->query('SELECT users.*, roles.name AS role_name FROM users LEFT JOIN roles ON users.role_id = roles.id ORDER BY users.id ASC');
    return $stmt->fetchAll();
}

/**
 * Obtiene la información de un usuario por su ID.
 *
 * @param PDO $pdo
 * @param int $id
 * @return array|null
 */
function getUserById(PDO $pdo, int $id): ?array
{
    $stmt = $pdo->prepare('SELECT users.*, roles.name AS role_name FROM users LEFT JOIN roles ON users.role_id = roles.id WHERE users.id = :id');
    $stmt->execute([':id' => $id]);
    $user = $stmt->fetch();
    return $user ?: null;
}

/**
 * Crea un usuario y lo asigna a un rol.
 *
 * @param PDO $pdo
 * @param string $username
 * @param string $password
 * @param int $roleId
 * @return int
 */
function createUser(PDO $pdo, string $username, string $password, int $roleId): int
{
    $hash = password_hash($password, PASSWORD_DEFAULT);
    $stmt = $pdo->prepare('INSERT INTO users (username, password, role_id) VALUES (:username, :password, :role_id)');
    $stmt->execute([
        ':username' => $username,
        ':password' => $hash,
        ':role_id'  => $roleId,
    ]);
    return (int)$pdo->lastInsertId();
}

/**
 * Actualiza un usuario y su rol. Si la contraseña es null se mantiene la actual.
 *
 * @param PDO $pdo
 * @param int $id
 * @param string $username
 * @param string|null $password
 * @param int $roleId
 */
function updateUser(PDO $pdo, int $id, string $username, ?string $password, int $roleId): void
{
    if ($password === null || $password === '') {
        $stmt = $pdo->prepare('UPDATE users SET username = :username, role_id = :role_id WHERE id = :id');
        $stmt->execute([
            ':username' => $username,
            ':role_id'  => $roleId,
            ':id'       => $id,
        ]);
    } else {
        $hash = password_hash($password, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare('UPDATE users SET username = :username, password = :password, role_id = :role_id WHERE id = :id');
        $stmt->execute([
            ':username' => $username,
            ':password' => $hash,
            ':role_id'  => $roleId,
            ':id'       => $id,
        ]);
    }
}

/**
 * Elimina un usuario. No permite eliminar tu propia cuenta.
 *
 * @param PDO $pdo
 * @param int $id
 */
function deleteUser(PDO $pdo, int $id): void
{
    if (!isLoggedIn() || $_SESSION['user_id'] == $id) {
        return;
    }
    $stmt = $pdo->prepare('DELETE FROM users WHERE id = :id');
    $stmt->execute([':id' => $id]);
}

/**
 * Comprueba si el usuario actual tiene permiso sobre un módulo y acción.
 *
 * @param string $module
 * @param string $action view|create|edit|delete
 * @return bool
 */
function hasPermission(string $module, string $action = 'view'): bool
{
    $user = currentUser();
    if (!$user) {
        return false;
    }
    // Los superadministradores siempre tienen permiso
    if (isset($user['role']) && strtolower($user['role']) === 'superadmin') {
        return true;
    }
    $pdo = getPDO();
    if (!$pdo) {
        return false;
    }
    // Asegurar tablas de roles
    ensureRoleTables($pdo);
    // Obtener role_id
    $roleId = $user['role_id'] ?? null;
    if (!$roleId) {
        // Si no tiene role_id, buscar por el nombre del rol
        $roleName = $user['role'] ?? '';
        if ($roleName) {
            $stmt = $pdo->prepare('SELECT id FROM roles WHERE name = :name');
            $stmt->execute([':name' => $roleName]);
            $roleId = $stmt->fetchColumn();
        }
    }
    if (!$roleId) {
        return false;
    }
    // Consultar permisos
    $stmt = $pdo->prepare('SELECT can_view, can_create, can_edit, can_delete FROM role_permissions WHERE role_id = :role_id AND module = :module');
    $stmt->execute([
        ':role_id' => $roleId,
        ':module'  => $module,
    ]);
    $row = $stmt->fetch();
    if (!$row) {
        return false;
    }
    switch ($action) {
        case 'view':
            return (bool)$row['can_view'];
        case 'create':
            return (bool)$row['can_create'];
        case 'edit':
            return (bool)$row['can_edit'];
        case 'delete':
            return (bool)$row['can_delete'];
        default:
            return false;
    }
}

/**
 * Requiere que el usuario tenga permiso para acceder al módulo y acción especificados.
 * Si no lo tiene, redirige a la página principal o muestra un mensaje.
 *
 * @param string $module
 * @param string $action
 */
function requirePermission(string $module, string $action = 'view'): void
{
    if (!hasPermission($module, $action)) {
        // Mostrar mensaje o redirigir
        echo '<div class="container mt-5"><div class="alert alert-danger">No tiene permiso para acceder a esta sección.</div></div>';
        exit;
    }
}

/**
 * Asegura la existencia de las tablas de cajas y sesiones de caja.
 *
 * @param PDO $pdo
 */
function ensureBoxTables(PDO $pdo): void
{
    // Tabla de cajas
    $pdo->exec(
        "CREATE TABLE IF NOT EXISTS boxes (\n" .
        "    id INT AUTO_INCREMENT PRIMARY KEY,\n" .
        "    name VARCHAR(100) NOT NULL UNIQUE,\n" .
        "    description VARCHAR(255) NULL,\n" .
        "    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP\n" .
        ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;"
    );
    // Sesiones de caja
    $pdo->exec(
        "CREATE TABLE IF NOT EXISTS cash_sessions (\n" .
        "    id INT AUTO_INCREMENT PRIMARY KEY,\n" .
        "    box_id INT NOT NULL,\n" .
        "    user_id INT NOT NULL,\n" .
        "    opened_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,\n" .
        "    initial_amount DECIMAL(15,2) NOT NULL,\n" .
        "    closed_at TIMESTAMP NULL DEFAULT NULL,\n" .
        "    closed_by INT NULL,\n" .
        "    final_amount DECIMAL(15,2) NULL,\n" .
        "    difference DECIMAL(15,2) NULL,\n" .
        "    CONSTRAINT fk_sessions_box FOREIGN KEY (box_id) REFERENCES boxes(id) ON DELETE CASCADE,\n" .
        "    CONSTRAINT fk_sessions_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE\n" .
        ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;"
    );
    // Movimientos de caja
    $pdo->exec(
        "CREATE TABLE IF NOT EXISTS cash_movements (\n" .
        "    id INT AUTO_INCREMENT PRIMARY KEY,\n" .
        "    session_id INT NOT NULL,\n" .
        "    user_id INT NOT NULL,\n" .
        "    movement_type ENUM('ingreso','retiro') NOT NULL,\n" .
        "    amount DECIMAL(15,2) NOT NULL,\n" .
        "    description VARCHAR(255) NULL,\n" .
        "    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,\n" .
        "    CONSTRAINT fk_movements_session FOREIGN KEY (session_id) REFERENCES cash_sessions(id) ON DELETE CASCADE,\n" .
        "    CONSTRAINT fk_movements_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE\n" .
        ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;"
    );
}

/**
 * Asegura la existencia de la tabla de medios de pago.
 *
 * Cada medio de pago tiene un nombre, un estado de activación y un porcentaje de descuento.
 *
 * @param PDO $pdo
 */
function ensurePaymentTables(PDO $pdo): void
{
    $pdo->exec(
        "CREATE TABLE IF NOT EXISTS payment_methods (\n" .
        "    id INT AUTO_INCREMENT PRIMARY KEY,\n" .
        "    name VARCHAR(100) NOT NULL UNIQUE,\n" .
        "    active TINYINT(1) NOT NULL DEFAULT 1,\n" .
        "    discount_percent DECIMAL(5,2) NOT NULL DEFAULT 0,\n" .
        "    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP\n" .
        ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;"
    );
}

/**
 * Obtiene todos los medios de pago.
 *
 * @param PDO $pdo
 * @return array
 */
function getPaymentMethods(PDO $pdo): array
{
    $stmt = $pdo->query('SELECT * FROM payment_methods ORDER BY id ASC');
    return $stmt->fetchAll();
}

/**
 * Obtiene los datos de un medio de pago por ID.
 *
 * @param PDO $pdo
 * @param int $id
 * @return array|null
 */
function getPaymentMethodById(PDO $pdo, int $id): ?array
{
    $stmt = $pdo->prepare('SELECT * FROM payment_methods WHERE id = :id');
    $stmt->execute([':id' => $id]);
    $pm = $stmt->fetch();
    return $pm ?: null;
}

/**
 * Crea un nuevo medio de pago.
 *
 * @param PDO $pdo
 * @param string $name
 * @param bool $active
 * @param float $discountPercent
 */
function createPaymentMethod(PDO $pdo, string $name, bool $active, float $discountPercent): void
{
    $stmt = $pdo->prepare('INSERT INTO payment_methods (name, active, discount_percent) VALUES (:name, :active, :discount)');
    $stmt->execute([
        ':name'    => $name,
        ':active'  => $active ? 1 : 0,
        ':discount' => $discountPercent,
    ]);
}

/**
 * Actualiza un medio de pago existente.
 *
 * @param PDO $pdo
 * @param int $id
 * @param string $name
 * @param bool $active
 * @param float $discountPercent
 */
function updatePaymentMethod(PDO $pdo, int $id, string $name, bool $active, float $discountPercent): void
{
    $stmt = $pdo->prepare('UPDATE payment_methods SET name = :name, active = :active, discount_percent = :discount WHERE id = :id');
    $stmt->execute([
        ':name'    => $name,
        ':active'  => $active ? 1 : 0,
        ':discount' => $discountPercent,
        ':id'      => $id,
    ]);
}

/**
 * Elimina un medio de pago.
 *
 * @param PDO $pdo
 * @param int $id
 */
function deletePaymentMethod(PDO $pdo, int $id): void
{
    $stmt = $pdo->prepare('DELETE FROM payment_methods WHERE id = :id');
    $stmt->execute([':id' => $id]);
}

/**
 * Devuelve todas las cajas.
 *
 * @param PDO $pdo
 * @return array
 */
function getBoxes(PDO $pdo): array
{
    $stmt = $pdo->query('SELECT * FROM boxes ORDER BY id ASC');
    return $stmt->fetchAll();
}

/**
 * Devuelve los datos de una caja.
 *
 * @param PDO $pdo
 * @param int $id
 * @return array|null
 */
function getBoxById(PDO $pdo, int $id): ?array
{
    $stmt = $pdo->prepare('SELECT * FROM boxes WHERE id = :id');
    $stmt->execute([':id' => $id]);
    $box = $stmt->fetch();
    return $box ?: null;
}

/**
 * Crea una nueva caja.
 *
 * @param PDO $pdo
 * @param string $name
 * @param string $description
 */
function createBox(PDO $pdo, string $name, string $description): void
{
    $stmt = $pdo->prepare('INSERT INTO boxes (name, description) VALUES (:name, :description)');
    $stmt->execute([':name' => $name, ':description' => $description]);
}

/**
 * Actualiza una caja.
 *
 * @param PDO $pdo
 * @param int $id
 * @param string $name
 * @param string $description
 */
function updateBox(PDO $pdo, int $id, string $name, string $description): void
{
    $stmt = $pdo->prepare('UPDATE boxes SET name = :name, description = :description WHERE id = :id');
    $stmt->execute([':name' => $name, ':description' => $description, ':id' => $id]);
}

/**
 * Elimina una caja si no tiene sesión abierta.
 *
 * @param PDO $pdo
 * @param int $id
 */
function deleteBox(PDO $pdo, int $id): void
{
    // Comprobar si hay sesión abierta
    $open = getOpenSession($pdo, $id);
    if ($open) {
        return;
    }
    $stmt = $pdo->prepare('DELETE FROM boxes WHERE id = :id');
    $stmt->execute([':id' => $id]);
}

/**
 * Obtiene la sesión abierta de una caja (si existe).
 *
 * @param PDO $pdo
 * @param int $boxId
 * @return array|null
 */
function getOpenSession(PDO $pdo, int $boxId): ?array
{
    $stmt = $pdo->prepare('SELECT * FROM cash_sessions WHERE box_id = :box_id AND closed_at IS NULL LIMIT 1');
    $stmt->execute([':box_id' => $boxId]);
    $session = $stmt->fetch();
    return $session ?: null;
}

/**
 * Abre una sesión de caja.
 *
 * @param PDO $pdo
 * @param int $boxId
 * @param int $userId
 * @param float $initialAmount
 */
function openBoxSession(PDO $pdo, int $boxId, int $userId, float $initialAmount): void
{
    // Verificar que no exista otra sesión abierta
    if (getOpenSession($pdo, $boxId)) {
        return;
    }
    $stmt = $pdo->prepare('INSERT INTO cash_sessions (box_id, user_id, initial_amount) VALUES (:box_id, :user_id, :initial_amount)');
    $stmt->execute([
        ':box_id'       => $boxId,
        ':user_id'      => $userId,
        ':initial_amount' => $initialAmount,
    ]);
}

/**
 * Registra un movimiento de caja (ingreso o retiro).
 *
 * @param PDO $pdo
 * @param int $sessionId
 * @param string $type 'ingreso' o 'retiro'
 * @param float $amount
 * @param string $description
 * @param int $userId
 */
function addCashMovement(PDO $pdo, int $sessionId, string $type, float $amount, string $description, int $userId): void
{
    $stmt = $pdo->prepare('INSERT INTO cash_movements (session_id, user_id, movement_type, amount, description) VALUES (:session, :user, :type, :amount, :description)');
    $stmt->execute([
        ':session'     => $sessionId,
        ':user'        => $userId,
        ':type'        => $type,
        ':amount'      => $amount,
        ':description' => $description,
    ]);
}

/**
 * Cierra una sesión de caja calculando el saldo esperado y la diferencia.
 *
 * @param PDO $pdo
 * @param int $sessionId
 * @param int $userId Usuario que cierra la caja
 * @param float $finalAmount
 */
function closeBoxSession(PDO $pdo, int $sessionId, int $userId, float $finalAmount): void
{
    // Obtener datos de la sesión
    $stmt = $pdo->prepare('SELECT * FROM cash_sessions WHERE id = :id');
    $stmt->execute([':id' => $sessionId]);
    $session = $stmt->fetch();
    if (!$session || $session['closed_at'] !== null) {
        return;
    }
    // Calcular saldo esperado
    $expected = (float)$session['initial_amount'];
    // Total ingresos y retiros
    $stmtSum = $pdo->prepare('SELECT SUM(CASE WHEN movement_type = "ingreso" THEN amount ELSE 0 END) AS ingresos, SUM(CASE WHEN movement_type = "retiro" THEN amount ELSE 0 END) AS retiros FROM cash_movements WHERE session_id = :session_id');
    $stmtSum->execute([':session_id' => $sessionId]);
    $sums = $stmtSum->fetch();
    $ingresos = (float)$sums['ingresos'];
    $retiros  = (float)$sums['retiros'];
    $expected = $expected + $ingresos - $retiros;
    $difference = $finalAmount - $expected;
    // Actualizar sesión
    $stmtClose = $pdo->prepare('UPDATE cash_sessions SET closed_at = NOW(), closed_by = :closed_by, final_amount = :final, difference = :diff WHERE id = :id');
    $stmtClose->execute([
        ':closed_by' => $userId,
        ':final'     => $finalAmount,
        ':diff'      => $difference,
        ':id'        => $sessionId,
    ]);
}

/**
 * Obtiene los movimientos de una sesión.
 *
 * @param PDO $pdo
 * @param int $sessionId
 * @return array
 */
function getSessionMovements(PDO $pdo, int $sessionId): array
{
    $stmt = $pdo->prepare('SELECT cm.*, u.username FROM cash_movements cm LEFT JOIN users u ON cm.user_id = u.id WHERE cm.session_id = :id ORDER BY cm.id ASC');
    $stmt->execute([':id' => $sessionId]);
    return $stmt->fetchAll();
}

/**
 * Obtiene resumen de una sesión de caja.
 *
 * @param PDO $pdo
 * @param int $sessionId
 * @return array|null
 */
function getSessionSummary(PDO $pdo, int $sessionId): ?array
{
    $stmt = $pdo->prepare('SELECT * FROM cash_sessions WHERE id = :id');
    $stmt->execute([':id' => $sessionId]);
    $session = $stmt->fetch();
    if (!$session) {
        return null;
    }
    // Totales
    $sumStmt = $pdo->prepare('SELECT SUM(CASE WHEN movement_type = "ingreso" THEN amount ELSE 0 END) AS ingresos, SUM(CASE WHEN movement_type = "retiro" THEN amount ELSE 0 END) AS retiros FROM cash_movements WHERE session_id = :session');
    $sumStmt->execute([':session' => $sessionId]);
    $sum = $sumStmt->fetch();
    $ingresos = (float)$sum['ingresos'];
    $retiros  = (float)$sum['retiros'];
    $expected = (float)$session['initial_amount'] + $ingresos - $retiros;
    return [
        'session' => $session,
        'ingresos' => $ingresos,
        'retiros'  => $retiros,
        'expected' => $expected,
        'difference' => $session['difference'],
    ];
}

/**
 * Obtiene el historial de sesiones cerradas para una caja.
 *
 * @param PDO $pdo
 * @param int $boxId
 * @return array
 */
function getSessionHistory(PDO $pdo, int $boxId): array
{
    $stmt = $pdo->prepare('SELECT cs.*, u.username AS opened_by, cu.username AS closed_by_name FROM cash_sessions cs LEFT JOIN users u ON cs.user_id = u.id LEFT JOIN users cu ON cs.closed_by = cu.id WHERE cs.box_id = :box_id AND cs.closed_at IS NOT NULL ORDER BY cs.id DESC');
    $stmt->execute([':box_id' => $boxId]);
    return $stmt->fetchAll();
}

/**
 * Inserta el usuario SuperAdmin en la base de datos.
 *
 * @param PDO $pdo
 * @param string $username
 * @param string $password
 */
function insertSuperAdmin(PDO $pdo, string $username, string $password): void
{
    $hash = password_hash($password, PASSWORD_DEFAULT);
    $stmt = $pdo->prepare('INSERT INTO users (username, password, role) VALUES (:u, :p, :r)');
    $stmt->execute([
        ':u' => $username,
        ':p' => $hash,
        ':r' => 'superadmin',
    ]);
}

/**
 * Guarda una clave de configuración en la tabla settings. Si ya existe, la actualiza.
 *
 * @param PDO $pdo
 * @param string $name
 * @param string $value
 */
function saveSetting(PDO $pdo, string $name, string $value): void
{
    $stmt = $pdo->prepare('INSERT INTO settings (name, value) VALUES (:n, :v) ON DUPLICATE KEY UPDATE value = VALUES(value)');
    $stmt->execute([
        ':n' => $name,
        ':v' => $value,
    ]);
}

/**
 * Obtiene el valor de una configuración por su nombre.
 *
 * @param PDO $pdo
 * @param string $name
 * @return string|null
 */
function getSetting(PDO $pdo, string $name): ?string
{
    $stmt = $pdo->prepare('SELECT value FROM settings WHERE name = :n');
    $stmt->execute([':n' => $name]);
    $row = $stmt->fetch();
    return $row['value'] ?? null;
}

/**
 * Guarda el archivo de configuración con los parámetros proporcionados.
 *
 * @param array $config
 * @param string $path
 * @throws RuntimeException
 */
function saveConfigFile(array $config, string $path): void
{
    $export = var_export($config, true);
    $content = "<?php\nreturn " . $export . ";\n";
    if (file_put_contents($path, $content) === false) {
        throw new RuntimeException('No se pudo guardar el archivo de configuración.');
    }
    // Proteger el archivo contra lectura por el navegador (requiere .htaccess en servidores Apache)
}