<?php
/*
 * Gestión de cajas y sesiones de caja.
 * Se incluye el CRUD de cajas, la apertura/cierre de sesión de caja, registro de movimientos
 * de ingreso/retiro y visualización de historial y detalles de sesiones.
 */

require_once __DIR__ . '/includes/functions.php';
requireLogin();
requirePermission('boxes', 'view');

$pdo = getPDO();
if (!$pdo) {
    echo '<div class="container mt-5"><div class="alert alert-danger">No se pudo conectar a la base de datos.</div></div>';
    exit;
}
// Asegurar tablas
ensureRoleTables($pdo);
ensureBoxTables($pdo);

$action = $_GET['action'] ?? 'list';
$id     = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$sessionId = isset($_GET['session_id']) ? (int)$_GET['session_id'] : 0;

// Control de acciones
// Delete box
if ($action === 'delete' && $id > 0) {
    requirePermission('boxes', 'delete');
    deleteBox($pdo, $id);
    redirect('boxes.php');
}
// Close session
if ($action === 'close' && $sessionId > 0 && $_SERVER['REQUEST_METHOD'] === 'POST') {
    requirePermission('boxes','edit');
    $finalAmount = (float)($_POST['final_amount'] ?? 0);
    closeBoxSession($pdo, $sessionId, (int)$_SESSION['user_id'], $finalAmount);
    // After closing, redirect to box list
    redirect('boxes.php?action=list');
}
// Add movement in session
if ($action === 'add-movement' && $sessionId > 0 && $_SERVER['REQUEST_METHOD'] === 'POST') {
    requirePermission('boxes', 'edit');
    $type  = $_POST['movement_type'] ?? '';
    $amount = (float)($_POST['amount'] ?? 0);
    $desc  = trim($_POST['description'] ?? '');
    if (in_array($type, ['ingreso','retiro']) && $amount > 0) {
        addCashMovement($pdo, $sessionId, $type, $amount, $desc, (int)$_SESSION['user_id']);
    }
    redirect('boxes.php?action=manage&id=' . (int)$id);
}
// Process create/edit box
if (($_SERVER['REQUEST_METHOD'] === 'POST') && ($action === 'create' || ($action === 'edit' && $id > 0))) {
    $name        = trim($_POST['name'] ?? '');
    $description = trim($_POST['description'] ?? '');
    if ($name !== '') {
        if ($action === 'create') {
            requirePermission('boxes','create');
            // Verificar nombre único
            $stmt = $pdo->prepare('SELECT id FROM boxes WHERE name = :n');
            $stmt->execute([':n' => $name]);
            if ($stmt->fetch()) {
                $errorMsg = 'Ya existe una caja con ese nombre.';
            } else {
                createBox($pdo, $name, $description);
                redirect('boxes.php');
            }
        } else {
            requirePermission('boxes','edit');
            updateBox($pdo, $id, $name, $description);
            redirect('boxes.php');
        }
    } else {
        $errorMsg = 'El nombre es obligatorio.';
    }
}
// Open session form (POST)
if ($action === 'open' && $id > 0 && $_SERVER['REQUEST_METHOD'] === 'POST') {
    requirePermission('boxes','edit');
    $initialAmount = (float)($_POST['initial_amount'] ?? 0);
    if ($initialAmount >= 0) {
        openBoxSession($pdo, $id, (int)$_SESSION['user_id'], $initialAmount);
        redirect('boxes.php?action=manage&id=' . $id);
    } else {
        $errorMsg = 'El monto inicial no puede ser negativo.';
    }
}

require_once __DIR__ . '/layout.php';
renderHeader('Cajas', 'boxes');

?>
<div class="container-fluid">
    <?php if ($action === 'create' || ($action === 'edit' && $id > 0)): ?>
        <?php
        $editing = ($action === 'edit');
        $box = ['name' => '', 'description' => ''];
        if ($editing) {
            $boxData = getBoxById($pdo, $id);
            if (!$boxData) {
                echo '<div class="alert alert-danger">Caja no encontrada.</div>';
            } else {
                $box = $boxData;
            }
        }
        ?>
        <h1 class="mb-4"><?php echo $editing ? 'Editar caja' : 'Crear caja'; ?></h1>
        <?php if (!empty($errorMsg)): ?>
            <div class="alert alert-danger"><?php echo escape($errorMsg); ?></div>
        <?php endif; ?>
        <form method="post" action="">
            <div class="mb-3">
                <label for="name" class="form-label">Nombre</label>
                <input type="text" class="form-control" id="name" name="name" required value="<?php echo escape($box['name']); ?>">
            </div>
            <div class="mb-3">
                <label for="description" class="form-label">Descripción</label>
                <input type="text" class="form-control" id="description" name="description" value="<?php echo escape($box['description']); ?>">
            </div>
            <button type="submit" class="btn btn-primary"><?php echo $editing ? 'Actualizar' : 'Crear'; ?></button>
            <a href="boxes.php" class="btn btn-secondary">Cancelar</a>
        </form>
    <?php elseif ($action === 'manage' && $id > 0): ?>
        <?php
        $box = getBoxById($pdo, $id);
        if (!$box) {
            echo '<div class="alert alert-danger">Caja no encontrada.</div>';
        } else {
            $openSession = getOpenSession($pdo, $box['id']);
        ?>
            <h1 class="mb-3">Gestión de Caja: <?php echo escape($box['name']); ?></h1>
            <?php if (!$openSession): ?>
                <p>No hay sesión abierta para esta caja.</p>
                <?php if (hasPermission('boxes','edit')): ?>
                    <form method="post" action="boxes.php?action=open&id=<?php echo $box['id']; ?>">
                        <div class="mb-3">
                            <label for="initial_amount" class="form-label">Monto inicial</label>
                            <input type="number" step="0.01" min="0" class="form-control" id="initial_amount" name="initial_amount" required>
                        </div>
                        <button type="submit" class="btn btn-success">Abrir caja</button>
                        <a href="boxes.php" class="btn btn-secondary">Volver</a>
                    </form>
                <?php endif; ?>
            <?php else: ?>
                <?php
                $sessionId = $openSession['id'];
                $summary = getSessionSummary($pdo, $sessionId);
                ?>
                <h5>Sesión abierta: #<?php echo (int)$sessionId; ?> - Iniciada el <?php echo escape($openSession['opened_at']); ?></h5>
                <div class="row">
                    <div class="col-lg-4">
                        <div class="card mb-3">
                            <div class="card-header">Resumen</div>
                            <div class="card-body">
                                <p><strong>Monto inicial:</strong> <?php echo number_format($summary['session']['initial_amount'], 2); ?> €</p>
                                <p><strong>Ingresos:</strong> <?php echo number_format($summary['ingresos'], 2); ?> €</p>
                                <p><strong>Retiros:</strong> <?php echo number_format($summary['retiros'], 2); ?> €</p>
                                <p><strong>Saldo esperado:</strong> <?php echo number_format($summary['expected'], 2); ?> €</p>
                            </div>
                        </div>
                        <?php if (hasPermission('boxes','edit')): ?>
                        <div class="card mb-3">
                            <div class="card-header">Cerrar caja</div>
                            <div class="card-body">
                                <form method="post" action="boxes.php?action=close&session_id=<?php echo $sessionId; ?>">
                                    <div class="mb-3">
                                        <label for="final_amount" class="form-label">Monto final contado</label>
                                        <input type="number" step="0.01" min="0" class="form-control" id="final_amount" name="final_amount" required>
                                    </div>
                                    <button type="submit" class="btn btn-danger">Cerrar caja</button>
                                </form>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>
                    <div class="col-lg-8">
                        <div class="card mb-3">
                            <div class="card-header">Movimientos</div>
                            <div class="card-body">
                                <?php if (hasPermission('boxes','edit')): ?>
                                <form method="post" action="boxes.php?action=add-movement&session_id=<?php echo $sessionId; ?>&id=<?php echo $box['id']; ?>" class="row g-2 align-items-end mb-3">
                                    <div class="col-md-3">
                                        <label for="movement_type" class="form-label">Tipo</label>
                                        <select name="movement_type" id="movement_type" class="form-select" required>
                                            <option value="ingreso">Ingreso</option>
                                            <option value="retiro">Retiro</option>
                                        </select>
                                    </div>
                                    <div class="col-md-3">
                                        <label for="amount" class="form-label">Monto</label>
                                        <input type="number" step="0.01" min="0.01" name="amount" id="amount" class="form-control" required>
                                    </div>
                                    <div class="col-md-4">
                                        <label for="description" class="form-label">Descripción</label>
                                        <input type="text" name="description" id="description" class="form-control">
                                    </div>
                                    <div class="col-md-2">
                                        <button type="submit" class="btn btn-primary w-100">Registrar</button>
                                    </div>
                                </form>
                                <?php endif; ?>
                                <?php $movs = getSessionMovements($pdo, $sessionId); ?>
                                <div class="table-responsive">
                                    <table class="table table-striped table-bordered">
                                        <thead>
                                        <tr>
                                            <th>ID</th>
                                            <th>Fecha</th>
                                            <th>Tipo</th>
                                            <th>Monto</th>
                                            <th>Usuario</th>
                                            <th>Descripción</th>
                                        </tr>
                                        </thead>
                                        <tbody>
                                        <?php foreach ($movs as $m): ?>
                                            <tr>
                                                <td><?php echo (int)$m['id']; ?></td>
                                                <td><?php echo escape($m['created_at']); ?></td>
                                                <td><?php echo ucfirst($m['movement_type']); ?></td>
                                                <td><?php echo number_format($m['amount'], 2); ?> €</td>
                                                <td><?php echo escape($m['username']); ?></td>
                                                <td><?php echo escape($m['description']); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <a href="boxes.php" class="btn btn-secondary">Volver</a>
            <?php endif; ?>
        <?php }
        ?>
    <?php elseif ($action === 'history' && $id > 0): ?>
        <?php $box = getBoxById($pdo, $id); ?>
        <?php if (!$box): ?>
            <div class="alert alert-danger">Caja no encontrada.</div>
        <?php else: ?>
            <h1 class="mb-3">Historial de cierres: <?php echo escape($box['name']); ?></h1>
            <?php $history = getSessionHistory($pdo, $id); ?>
            <div class="table-responsive">
                <table class="table table-striped table-bordered">
                    <thead>
                    <tr>
                        <th>ID Sesión</th>
                        <th>Abierta por</th>
                        <th>Fecha apertura</th>
                        <th>Fecha cierre</th>
                        <th>Inicial</th>
                        <th>Final</th>
                        <th>Diferencia</th>
                        <th>Acciones</th>
                    </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($history as $h): ?>
                        <tr>
                            <td><?php echo (int)$h['id']; ?></td>
                            <td><?php echo escape($h['opened_by']); ?></td>
                            <td><?php echo escape($h['opened_at']); ?></td>
                            <td><?php echo escape($h['closed_at']); ?></td>
                            <td><?php echo number_format($h['initial_amount'], 2); ?> €</td>
                            <td><?php echo number_format($h['final_amount'], 2); ?> €</td>
                            <td><?php echo number_format($h['difference'], 2); ?> €</td>
                            <td><a href="boxes.php?action=detail&session_id=<?php echo (int)$h['id']; ?>" class="btn btn-sm btn-info">Detalle</a></td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <a href="boxes.php" class="btn btn-secondary">Volver</a>
        <?php endif; ?>
    <?php elseif ($action === 'detail' && $sessionId > 0): ?>
        <?php $summary = getSessionSummary($pdo, $sessionId); ?>
        <?php if (!$summary): ?>
            <div class="alert alert-danger">Sesión no encontrada.</div>
        <?php else: ?>
            <?php $session = $summary['session']; ?>
            <?php $movs = getSessionMovements($pdo, $sessionId); ?>
            <h1 class="mb-3">Detalle de sesión #<?php echo (int)$sessionId; ?></h1>
            <div class="row">
                <div class="col-lg-4">
                    <div class="card mb-3">
                        <div class="card-header">Resumen</div>
                        <div class="card-body">
                            <p><strong>Caja:</strong> <?php echo escape(getBoxById($pdo, $session['box_id'])['name'] ?? ''); ?></p>
                            <p><strong>Abierta por:</strong> <?php $opener = getUserById($pdo, $session['user_id']); echo escape($opener['username'] ?? ''); ?></p>
                            <p><strong>Fecha apertura:</strong> <?php echo escape($session['opened_at']); ?></p>
                            <p><strong>Fecha cierre:</strong> <?php echo escape($session['closed_at']); ?></p>
                            <p><strong>Monto inicial:</strong> <?php echo number_format($session['initial_amount'], 2); ?> €</p>
                            <p><strong>Ingresos:</strong> <?php echo number_format($summary['ingresos'], 2); ?> €</p>
                            <p><strong>Retiros:</strong> <?php echo number_format($summary['retiros'], 2); ?> €</p>
                            <p><strong>Saldo esperado:</strong> <?php echo number_format($summary['expected'], 2); ?> €</p>
                            <p><strong>Monto final:</strong> <?php echo number_format($session['final_amount'], 2); ?> €</p>
                            <p><strong>Diferencia:</strong> <?php echo number_format($session['difference'], 2); ?> €</p>
                        </div>
                    </div>
                </div>
                <div class="col-lg-8">
                    <div class="card mb-3">
                        <div class="card-header">Movimientos</div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-striped table-bordered">
                                    <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Fecha</th>
                                        <th>Tipo</th>
                                        <th>Monto</th>
                                        <th>Usuario</th>
                                        <th>Descripción</th>
                                    </tr>
                                    </thead>
                                    <tbody>
                                    <?php foreach ($movs as $m): ?>
                                        <tr>
                                            <td><?php echo (int)$m['id']; ?></td>
                                            <td><?php echo escape($m['created_at']); ?></td>
                                            <td><?php echo ucfirst($m['movement_type']); ?></td>
                                            <td><?php echo number_format($m['amount'], 2); ?> €</td>
                                            <td><?php echo escape($m['username']); ?></td>
                                            <td><?php echo escape($m['description']); ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <a href="boxes.php?action=history&id=<?php echo (int)$session['box_id']; ?>" class="btn btn-secondary">Volver</a>
        <?php endif; ?>
    <?php else: ?>
        <!-- Listado de cajas -->
        <h1 class="mb-4">Cajas</h1>
        <?php if (hasPermission('boxes','create')): ?>
            <a href="boxes.php?action=create" class="btn btn-success mb-3">Nueva caja</a>
        <?php endif; ?>
        <div class="table-responsive">
            <table class="table table-striped table-bordered">
                <thead>
                <tr>
                    <th>ID</th>
                    <th>Nombre</th>
                    <th>Descripción</th>
                    <th>Sesión</th>
                    <th>Acciones</th>
                </tr>
                </thead>
                <tbody>
                <?php foreach (getBoxes($pdo) as $b): ?>
                    <?php $session = getOpenSession($pdo, $b['id']); ?>
                    <tr>
                        <td><?php echo (int)$b['id']; ?></td>
                        <td><?php echo escape($b['name']); ?></td>
                        <td><?php echo escape($b['description']); ?></td>
                        <td>
                            <?php if ($session): ?>
                                <span class="badge bg-success">Abierta</span>
                            <?php else: ?>
                                <span class="badge bg-secondary">Cerrada</span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <?php if (hasPermission('boxes','edit')): ?>
                                <a href="boxes.php?action=manage&amp;id=<?php echo (int)$b['id']; ?>" class="btn btn-sm btn-info">Gestionar</a>
                            <?php endif; ?>
                            <a href="boxes.php?action=history&amp;id=<?php echo (int)$b['id']; ?>" class="btn btn-sm btn-secondary">Historial</a>
                            <?php if (hasPermission('boxes','edit')): ?>
                                <a href="boxes.php?action=edit&amp;id=<?php echo (int)$b['id']; ?>" class="btn btn-sm btn-primary">Editar</a>
                            <?php endif; ?>
                            <?php if (hasPermission('boxes','delete')): ?>
                                <?php if (!$session): ?>
                                    <a href="boxes.php?action=delete&amp;id=<?php echo (int)$b['id']; ?>" class="btn btn-sm btn-danger" onclick="return confirm('¿Seguro que desea eliminar esta caja?');">Eliminar</a>
                                <?php else: ?>
                                    <button class="btn btn-sm btn-danger" disabled>Eliminar</button>
                                <?php endif; ?>
                            <?php endif; ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>
</div>
<?php
renderFooter();