<?php
/*
 * Punto de Venta (Hito 5).
 *
 * Esta página implementa una interfaz responsiva para realizar ventas de forma fluida. Permite
 * buscar productos (lectura básica desde la BD de PrestaShop), agregarlos a un carrito local,
 * modificar cantidades, quitar productos, seleccionar un medio de pago y calcular el total
 * con el descuento correspondiente. Aún no se crean pedidos reales ni se descuenta el stock.
 */

require_once __DIR__ . '/includes/functions.php';
requireLogin();
requirePermission('pos', 'view');

$pdo = getPDO();
// Asegurar que existan los medios de pago y cargarlos
ensurePaymentTables($pdo);

// Inicializar carrito en sesión si no existe
if (!isset($_SESSION['cart']) || !is_array($_SESSION['cart'])) {
    $_SESSION['cart'] = [];
}

// Obtener lista de productos según búsqueda
$searchTerm = trim($_GET['search'] ?? '');
$products   = getProducts($searchTerm);

// Manejar acciones del carrito
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    // Añadir variante al carrito
    if ($action === 'add') {
        $key = trim($_POST['product_key'] ?? '');
        $variant = null;
        if ($key !== '') {
            $variant = getVariant($key);
        }
        $errorParam = '';
        if ($variant) {
            // Verificar stock disponible
            if ($variant['stock'] <= 0) {
                // Sin stock
                $errorParam = 'nostock';
            } else {
                // Calcular cantidad actual en carrito
                $currentQty = isset($_SESSION['cart'][$key]) ? (int)$_SESSION['cart'][$key]['qty'] : 0;
                if ($currentQty + 1 > $variant['stock']) {
                    // No alcanza el stock
                    $errorParam = 'stock';
                } else {
                    // Añadir o incrementar
                    if (isset($_SESSION['cart'][$key])) {
                        $_SESSION['cart'][$key]['qty'] += 1;
                    } else {
                        $_SESSION['cart'][$key] = [
                            'key'        => $variant['key'],
                            'product_id' => $variant['product_id'],
                            'attribute_id' => $variant['attribute_id'],
                            'name'       => $variant['name'],
                            'reference'  => $variant['reference'],
                            'price'      => $variant['price'],
                            'qty'        => 1,
                            'stock'      => $variant['stock'],
                        ];
                    }
                }
            }
        } else {
            // No se encontró variante
            $errorParam = 'notfound';
        }
        // Construir URL de redirección conservando la búsqueda y el error si corresponde
        $redirect = 'pos.php';
        $query = [];
        if ($searchTerm !== '') {
            $query['search'] = $searchTerm;
        }
        if ($errorParam !== '') {
            $query['error'] = $errorParam;
        }
        if ($query) {
            $redirect .= '?' . http_build_query($query);
        }
        redirect($redirect);
    }
    // Actualizar cantidades y medio de pago
    if ($action === 'update') {
        $errorParam = '';
        // Actualizar cantidades
        if (isset($_POST['quantities']) && is_array($_POST['quantities'])) {
            foreach ($_POST['quantities'] as $k => $qty) {
                $key = trim($k);
                $qty = (int)$qty;
                if (isset($_SESSION['cart'][$key])) {
                    if ($qty > 0) {
                        // Verificar stock disponible para esta variante
                        $variant = getVariant($key);
                        if ($variant) {
                            if ($qty > $variant['stock']) {
                                // Ajustar a máximo disponible
                                $_SESSION['cart'][$key]['qty'] = $variant['stock'];
                                $errorParam = 'stock';
                            } else {
                                $_SESSION['cart'][$key]['qty'] = $qty;
                            }
                        } else {
                            // Variante ya no existe, eliminar
                            unset($_SESSION['cart'][$key]);
                        }
                    } else {
                        unset($_SESSION['cart'][$key]);
                    }
                }
            }
        }
        // Actualizar método de pago seleccionado
        $selectedPayment = isset($_POST['payment_id']) ? (int)$_POST['payment_id'] : 0;
        $_SESSION['selected_payment_id'] = $selectedPayment;
        // Redirigir para evitar reenvío
        $redirect = 'pos.php';
        $query = [];
        if ($searchTerm !== '') {
            $query['search'] = $searchTerm;
        }
        if ($errorParam !== '') {
            $query['error'] = $errorParam;
        }
        if ($query) {
            $redirect .= '?' . http_build_query($query);
        }
        redirect($redirect);
    }
    // Quitar un producto
    if ($action === 'remove') {
        $key = trim($_POST['product_key'] ?? '');
        if ($key !== '' && isset($_SESSION['cart'][$key])) {
            unset($_SESSION['cart'][$key]);
        }
        $redirect = 'pos.php';
        $query = [];
        if ($searchTerm !== '') {
            $query['search'] = $searchTerm;
        }
        if ($query) {
            $redirect .= '?' . http_build_query($query);
        }
        redirect($redirect);
    }
    // Vaciar carrito
    if ($action === 'clear') {
        $_SESSION['cart'] = [];
        $_SESSION['selected_payment_id'] = 0;
        $redirect = 'pos.php';
        $query = [];
        if ($searchTerm !== '') {
            $query['search'] = $searchTerm;
        }
        $redirect .= ($query ? '?' . http_build_query($query) : '');
        redirect($redirect);
    }
    // Finalizar venta y generar pedido real
    if ($action === 'finish') {
        // Permiso de crear ventas (módulo pos, acción create)
        requirePermission('pos', 'create');
        $user = currentUser();
        if (!$user) {
            // No hay usuario logueado
            redirect('login.php');
        }
        // Verificar que hay productos en el carrito
        if (empty($cart)) {
            $redir = 'pos.php?error=empty';
            if ($searchTerm !== '') {
                $redir .= '&search=' . urlencode($searchTerm);
            }
            redirect($redir);
        }
        // Verificar que hay método de pago seleccionado
        if (!$selectedPayment) {
            $redir = 'pos.php?error=payment';
            if ($searchTerm !== '') {
                $redir .= '&search=' . urlencode($searchTerm);
            }
            redirect($redir);
        }
        // Construir array de items para el pedido
        $itemsForOrder = [];
        foreach ($cart as $item) {
            $itemsForOrder[] = [
                'product_id'   => $item['product_id'],
                'attribute_id' => $item['attribute_id'],
                'qty'          => $item['qty'],
                'name'         => $item['name'],
                'reference'    => $item['reference'],
                'price'        => $item['price'],
            ];
        }
        try {
            $orderId = createSaleAndOrder((int)$user['id'], $itemsForOrder, $selectedPayment, $subtotal, $discountAmount, $total);
            // Vaciar carrito tras crear pedido
            $_SESSION['cart'] = [];
            $_SESSION['selected_payment_id'] = 0;
            // Redirigir con mensaje de éxito y id de pedido
            $redirect = 'pos.php?msg=venta-real&order=' . $orderId;
            if ($searchTerm !== '') {
                $redirect .= '&search=' . urlencode($searchTerm);
            }
            redirect($redirect);
        } catch (Exception $ex) {
            // Registrar error en log y mostrar mensaje al usuario
            logError('Error al crear pedido: ' . $ex->getMessage());
            $redir = 'pos.php?error=order';
            if ($searchTerm !== '') {
                $redir .= '&search=' . urlencode($searchTerm);
            }
            redirect($redir);
        }
    }
}

// Obtener carrito y método de pago seleccionado
$cart = $_SESSION['cart'];
$selectedPaymentId = $_SESSION['selected_payment_id'] ?? 0;

// Calcular totales
$subtotal = 0;
foreach ($cart as $item) {
    $subtotal += $item['price'] * $item['qty'];
}
// Determinar descuento
$discountPercent = 0;
$selectedPayment = null;
if ($selectedPaymentId) {
    $pm = getPaymentMethodById($pdo, $selectedPaymentId);
    if ($pm && $pm['active']) {
        $discountPercent = (float)$pm['discount_percent'];
        $selectedPayment = $pm;
    }
}
$discountAmount = $subtotal * $discountPercent / 100;
$total = $subtotal - $discountAmount;

require_once __DIR__ . '/layout.php';
renderHeader('Punto de Venta', 'pos');
?>
<div class="container-fluid">
    <h1 class="mb-4">Punto de Venta</h1>
    <?php
    // Mensajes de éxito
    if (isset($_GET['msg'])) {
        if ($_GET['msg'] === 'venta') {
            // Mensaje de venta simulada (versión antigua)
            echo '<div class="alert alert-success">Venta registrada localmente (no se ha creado un pedido real).</div>';
        } elseif ($_GET['msg'] === 'venta-real') {
            $orderId = isset($_GET['order']) ? (int)$_GET['order'] : 0;
            $msg = 'Pedido creado correctamente en PrestaShop.';
            if ($orderId) {
                $msg .= ' ID de pedido: ' . $orderId . '.';
            }
            echo '<div class="alert alert-success">' . escape($msg) . '</div>';
        }
    }
    ?>
    <?php
    // Mensajes de error
    if (isset($_GET['error'])) {
        $errCode = $_GET['error'];
        $errMsg  = '';
        if ($errCode === 'nostock') {
            $errMsg = 'Producto sin stock disponible.';
        } elseif ($errCode === 'stock') {
            $errMsg = 'No hay stock suficiente para la cantidad solicitada.';
        } elseif ($errCode === 'notfound') {
            $errMsg = 'No se encontró la variante seleccionada.';
        } elseif ($errCode === 'empty') {
            $errMsg = 'El carrito está vacío. Agregue productos antes de finalizar.';
        } elseif ($errCode === 'payment') {
            $errMsg = 'Debe seleccionar un medio de pago válido para completar la venta.';
        } elseif ($errCode === 'order') {
            $errMsg = 'Ocurrió un error al crear el pedido en PrestaShop. Revise el log para más detalles.';
        }
        if ($errMsg !== '') {
            echo '<div class="alert alert-danger">' . escape($errMsg) . '</div>';
        }
    }
    ?>
    <div class="row">
        <!-- Listado de productos -->
        <div class="col-lg-8 mb-4">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">Productos</h5>
                    <form class="d-flex" method="get" action="pos.php">
                        <input type="text" name="search" class="form-control me-2" placeholder="Buscar..." value="<?php echo escape($searchTerm); ?>">
                        <button class="btn btn-outline-primary" type="submit">Buscar</button>
                    </form>
                </div>
                <div class="card-body p-2" style="min-height:400px; overflow-y:auto;">
                    <?php if (empty($products)): ?>
                        <p class="text-muted">No se encontraron productos.</p>
                    <?php else: ?>
                        <div class="row g-2">
                            <?php foreach ($products as $prod): ?>
                                <div class="col-6 col-sm-4 col-md-3 col-lg-3 col-xl-2 mb-3">
                                    <div class="product-card card h-100 p-2 d-flex flex-column">
                                        <div class="text-center icon"><i class="fas fa-box"></i></div>
                                        <h6 class="name mt-2 text-center" title="<?php echo escape($prod['name']); ?>"><?php echo escape($prod['name']); ?></h6>
                                        <p class="ref text-center mb-1"><?php echo escape($prod['reference']); ?></p>
                                        <div class="price-stock text-center mt-auto">
                                            <div class="fw-bold"><?php echo number_format($prod['price'], 2); ?> €</div>
                                            <div class="small text-muted">Stock: <?php echo (int)$prod['stock']; ?></div>
                                        </div>
                                        <form method="post" action="" class="mt-2">
                                            <input type="hidden" name="action" value="add">
                                            <input type="hidden" name="product_key" value="<?php echo escape($prod['id']); ?>">
                                            <?php if ((int)$prod['stock'] > 0): ?>
                                                <button type="submit" class="btn btn-primary btn-add w-100"><i class="fas fa-plus"></i> Agregar</button>
                                            <?php else: ?>
                                                <button type="button" class="btn btn-secondary btn-add w-100" disabled>Agotado</button>
                                            <?php endif; ?>
                                        </form>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <!-- Carrito -->
        <div class="col-lg-4 mb-4">
            <div class="card">
                <div class="card-header"><h5 class="mb-0">Carrito</h5></div>
                <div class="card-body">
                    <?php if (empty($cart)): ?>
                        <p class="text-muted">No hay productos en el carrito.</p>
                    <?php else: ?>
                        <form method="post" action="">
                            <input type="hidden" name="action" value="update">
                            <div class="table-responsive" style="max-height:300px;">
                                <table class="table table-sm align-middle">
                                    <thead>
                                    <tr>
                                        <th>Producto</th>
                                        <th class="text-center" style="width:70px;">Cant.</th>
                                        <th class="text-end">Precio</th>
                                        <th class="text-end">Total</th>
                                        <th></th>
                                    </tr>
                                    </thead>
                                    <tbody>
                                    <?php foreach ($cart as $id => $item): ?>
                                        <tr>
                                            <td><?php echo escape($item['name']); ?></td>
                                            <td class="text-center">
                                                <input type="number" min="1" class="form-control form-control-sm" name="quantities[<?php echo escape($id); ?>]" value="<?php echo (int)$item['qty']; ?>">
                                            </td>
                                            <td class="text-end"><?php echo number_format($item['price'], 2); ?> €</td>
                                            <td class="text-end"><?php echo number_format($item['price'] * $item['qty'], 2); ?> €</td>
                                            <td>
                                                <form method="post" action="" onsubmit="return confirm('¿Quitar producto?');">
                                                    <input type="hidden" name="action" value="remove">
                                                    <input type="hidden" name="product_key" value="<?php echo escape($id); ?>">
                                                    <button type="submit" class="btn btn-sm btn-link text-danger"><i class="fas fa-trash"></i></button>
                                                </form>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            <!-- Selección de medio de pago -->
                            <div class="mb-2">
                                <label for="payment_id" class="form-label">Medio de pago</label>
                                <select class="form-select" id="payment_id" name="payment_id">
                                    <option value="0">-- Seleccionar --</option>
                                    <?php foreach (getPaymentMethods($pdo) as $pm): ?>
                                        <?php if ($pm['active']): ?>
                                            <option value="<?php echo (int)$pm['id']; ?>" <?php echo ($selectedPaymentId == $pm['id']) ? 'selected' : ''; ?>><?php echo escape($pm['name']); ?> (<?php echo number_format($pm['discount_percent'], 2); ?>% de desc.)</option>
                                        <?php endif; ?>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="mb-2 d-flex justify-content-between">
                                <div>Subtotal:</div>
                                <div><?php echo number_format($subtotal, 2); ?> €</div>
                            </div>
                            <div class="mb-2 d-flex justify-content-between">
                                <div>Descuento (<?php echo number_format($discountPercent, 2); ?>%):</div>
                                <div>- <?php echo number_format($discountAmount, 2); ?> €</div>
                            </div>
                            <div class="mb-3 d-flex justify-content-between fs-5 fw-bold">
                                <div>Total:</div>
                                <div><?php echo number_format($total, 2); ?> €</div>
                            </div>
                            <div class="d-flex justify-content-between">
                                <button type="submit" class="btn btn-primary">Actualizar</button>
                                <form method="post" action="">
                                    <input type="hidden" name="action" value="clear">
                                    <button type="submit" class="btn btn-secondary" onclick="return confirm('¿Vaciar carrito?');">Vaciar</button>
                                </form>
                            </div>
                        </form>
                        <form method="post" action="" class="mt-3">
                            <input type="hidden" name="action" value="finish">
                            <button type="submit" class="btn btn-success w-100" <?php echo (empty($cart) || !$selectedPayment) ? 'disabled' : ''; ?>>Finalizar venta</button>
                        </form>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>
<?php renderFooter(); ?>